/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_debug.h"
 
/*
 * Internal definitions used within the MX library
 */

/* XXX - this needs to be right or match data needs to be a string */
/* we don't want to waste lots of time byte-swapping match data */
#define htonll(X) (X)
#define ntohll(X) (X)

#define MIN(A,B) ((A)<(B)?(A):(B))
#define MAX(A,B) ((A)>(B)?(A):(B))

/*
 * Grr - this is because I am lazy, and because I cannot decide exactly how
 * to do this.  The "fd_set" in mx_lib_endpoint requires this.  I could remove
 * this dependency by hiding the fd_set in a "struct select_stuff *" and
 * define struct select_stuff in another .h that only files whch need to 
 * access it would include, then they would include types.h.  Or maybe it's ok
 * to include types.h here, but it makes me queasy.
 */
#include <sys/types.h>
#include <sys/poll.h>

#include <pthread.h>

#include "mx__shim.h"

enum mx_post_type
  {
    MX_POST_TYPE_SEND,
    MX_POST_TYPE_RECV,
    MX_POST_TYPE_BCAST,
    MX_POST_TYPE_BARRIER
  };
typedef enum mx_post_type mx_post_type_t;

enum mx_send_recv_type
  {
    MX_SR_TYPE_SEND,
    MX_SR_TYPE_ISSEND,
    MX_SR_TYPE_ISSEND_ACK,
    MX_SR_TYPE_BARRIER,
    MX_SR_TYPE_BCAST,
    MX_SR_TYPE_BARRIER_ACK,
    MX_SR_TYPE_PUT,
    MX_SR_TYPE_PUT_ACK,
    MX_SR_TYPE_GET,
    MX_SR_TYPE_GET_DATA
  };
typedef enum mx_send_recv_type mx_sr_type_t;

/*
 * definition of a pending send
 */
struct mx_lib_send {
  uint32_t length;		/* send length */
    
  mx_segment_t *seg_list;	/* segment list */
  uint32_t seg_cnt;
    
  uint64_t match_info;
    
  mx_endpoint_addr_t dest;		/* destination address */

  uint32_t rdma_handle;              /* rdma handle */
  uint32_t rdma_offset;
  uint32_t rdma_status;
  mx_sr_type_t type;          /* type of send we're doing */
};


/*
 * definition of a pending receive
 */
struct mx_lib_recv {
  uint32_t length;		/* max length */

  mx_segment_t *seg_list;	/* segment list */
  uint32_t seg_cnt;

  uint64_t match_info;
  uint64_t match_mask;

  mx_sr_type_t type;          /* type of send we want to match */
};

/*
 * definition of a pending barrier
 */
struct mx_lib_barrier {
  uint64_t match_info;

  int nsock;			/* number of addresss in barrier */
  mx_sr_type_t type;          /* type for sanity */
};

/*
 * definition of a pending broadcast
 */
struct mx_lib_bcast {
  uint32_t length;		/* send length */

  mx_segment_t *seg_list;	/* segment list */
  uint32_t seg_cnt;

  uint64_t match_info;

  mx_endpoint_addr_t *dests;		/* destination address */
  uint32_t dest_count;
  uint32_t dest_skip;           /* my index, don't send to self */

  mx_sr_type_t type;          /* type of bcast we want to match */
};

/*
 * A generic "post" item.  This can describe a send, receive, barrier,
 * or broadcast
 */
struct mx_post {
  mx_post_type_t type;	  /* type of this post */
    
  struct mx_endpoint *endpoint; /* associated endpoint */
    
  int complete;		  /* set TRUE when this post completes */
  int buffered;
  pthread_cond_t wait_cond;
  pthread_cond_t buffered_cond;
  mx_status_t status;		  /* completion status */

  union {
    struct mx_lib_send send;
    struct mx_lib_recv recv;
    struct mx_lib_bcast bcast;
    struct mx_lib_barrier barrier;
  } ts;
    
  struct mx_post *next;	/* next on list */
  struct mx_post *prev;	/* previous on list */
};

union mx_request {
  struct mx_post post;
};
/*
 * descriptor for each known address
 */
struct mx_address_desc {
  mx_endpoint_addr_t address;
  int send_sock;		/* socket for sending to this address */
  int recv_sock;		/* socket for receiving from this address */

  struct mx_address_desc *next;
  struct mx_address_desc *prev;
};

/*
 * Structure describing a queued message to be received later
 */
struct mx_queued_msg {
  mx_endpoint_addr_t source;
  uint64_t match_info;
  mx_request_t sender_request;
  uint32_t length;
  mx_post_type_t type;
    
  struct mx_queued_msg *next;		/* forward and backward links */
  struct mx_queued_msg *prev;
    
  char data[8];		/* data of message will follow the structure */
  /* the actual allocated size for this struct will vary based */
  /* on the size of the queued message */
};
#define MX_QUEUED_MSG_SIZE(len) (sizeof (struct mx_queued_msg) - 8 + len)
                                    

/*
 * Structure defining a pending barrier in the library
 */
struct mx_pending_barrier {
  uint64_t match_info;
  mx_post_type_t type;
  uint32_t current_count;
  uint32_t total_count;

  struct mx_pending_barrier *next;
  struct mx_pending_barrier *prev;
};

struct mx_barrier_callback_tuple {
  void (*callback)(mx_endpoint_t endpoint, struct mx_post *post);
  struct mx_post* post;
};

#define MX_N_HNDL          1024
struct mx_handles {
  uint64_t ptr[MX_N_HNDL];
  int len[MX_N_HNDL];
  int flags[MX_N_HNDL];
};

/*
 * Structure defining a endpoint instance in the library
 */
struct mx_endpoint {
    
  struct mx_attr *attr_list;		/* list of attributes associated with */
					/* this endpoint */
    
  struct mx_post recv_list;	/* all pending receives on this endpoint */
  struct mx_post send_list;	/* all pending sends on this endpoint */
  struct mx_post bcast_list; /* all pending broadcasts on this endpoint */
  struct mx_post barrier_list; /* all pending barriers on this endpoint */
    
  struct mx_post completed_posts;  /* all completed posts */
  struct mx_post buffered_posts;
  struct mx_post putget_posts;

  struct mx_pending_barrier pending_barriers; /* barriers in progress */
  /* accessed only by the rx thread */

  struct mx_queued_msg queued_messages; /* queued unexpected messages */
  /* accessed by the main thread and the rx thread */

  struct mx_address_desc known_addresss;  /* all known remote addresss */
    
  pthread_cond_t peek_cond;
  pthread_cond_t probe_cond;
  
  /* RDMA */
  struct mx_handles handles;


  /* TCP specific stuff */
  mx_endpoint_addr_t my_address;		/* address associated with this endpoint */
  int socket;				/* socket we use to communicate */
  struct pollfd read_sock[FD_SETSIZE];
  int sock_count;
    
  /* next endpoint on global list */
  struct mx_endpoint *next;
  struct mx_endpoint *prev;
};

struct mx_group
{
  uint32_t addresss_count;
  char data[8];
};
#define MX_GROUP_SIZE(count) (sizeof (struct mx_group) - 8 + count*sizeof (mx_address_t))

/*
 * List of all our endpoints
 */
struct mx_endpoint *MX_endpoint_list;

//#define ENDPOINT_BASE 7000	/* base of IP endpoint numbers we will use */
extern int ENDPOINT_BASE;
#define MX_MAX_ENDPOINT 32	/* max number of endpoints */

enum mx_fdset_flags {
  MX_FDSET_NOT_CHANGED,
  MX_FDSET_NOT_INITIALIZED,
  MX_FDSET_CHANGED
};

enum mx_rdma_window_flag {
  MX_RDMA_WRITE = 1,
  MX_RDMA_READ = 2
};

/*
 * Internal function prototypes
 */

void mx_destroy_endpoint(struct mx_endpoint *);

/* post.c */
struct mx_post *mx_new_post(struct mx_endpoint *, mx_post_type_t, void *);
void                  mx_destroy_post(struct mx_post *);
void                  mx_destroy_post_list(struct mx_post *);
mx_endpoint_addr_t      *mx_new_address_list(mx_endpoint_addr_t *, uint32_t);
mx_segment_t       *mx_new_segment_list(mx_segment_t *, uint32_t);
void                  mx_destroy_segment_list(mx_segment_t *);
uint32_t              mx_segment_list_len(mx_segment_t *, uint32_t);
uint32_t              mx_wait_for_completions(struct mx_endpoint *,
                                              uint32_t);
void	              mx_poll_for_completions(struct mx_endpoint *);
void                  mx_run_asynch_rx(struct mx_endpoint *);
void                  mx_run_asynch_tx(struct mx_endpoint *);
uint32_t              mx_run_send_queue(struct mx_endpoint *);
uint32_t              mx_run_bcast_queue(struct mx_endpoint *);

mx_return_t mx_post_issend_ack(mx_endpoint_t endpoint, 
                               mx_endpoint_addr_t dst,
                               mx_request_t request);

mx_return_t           mx_post_put_ack(mx_endpoint_t endpoint, 
                                      mx_endpoint_addr_t dst,
                                      mx_request_t request,
                                      mx_status_code_t error);
void mx_signal_handler(int sig);

/* mx_irecv.c */
mx_return_t
mx_irecv_with_type(mx_endpoint_t endpoint,
                   mx_segment_t *segments_list,
                   uint32_t segments_count,
		   uint64_t match_info,
		   uint64_t match_mask,
                   void *callback_arg,
                   mx_request_t *request,
                   mx_sr_type_t type);

/* mx_ibcast.c */
struct mx_post * mx_create_bcast_post(struct mx_endpoint *pp,
				      mx_segment_t *segments_list,
				      uint32_t segments_count,
				      mx_endpoint_addr_t *addresss_list,
				      uint32_t addresss_count,
				      uint32_t root_index,
				      uint64_t match_info,
				      void* context,
				      mx_sr_type_t type);
mx_return_t mx_ibcast_with_type(mx_endpoint_t endpoint,
                                mx_segment_t *segments_list,
                                uint32_t segments_count,
                                mx_endpoint_addr_t *addresss_list,
                                uint32_t addresss_count,
                                uint32_t root_index,
				uint64_t match_info,
                                void *context,
                                mx_request_t *request,
                                mx_sr_type_t type);
uint32_t mx_try_bcast(struct mx_post *);

/* mx_ibarrier.c */
void mx_barrier_callback (mx_endpoint_t endpoint, struct mx_post *post);

/* mx_isend.c */
mx_return_t mx_isend_with_type(mx_endpoint_t endpoint, 
                               mx_segment_t *segments_list, 
                               uint32_t segments_count, 
                               mx_endpoint_addr_t destination, 
			       uint64_t match_info,
                               void *context, 
                               mx_request_t *request,
                               mx_sr_type_t type,
                               uint32_t rdma_handle,
                               uint32_t rdma_offset,
                               uint32_t rmda_status);
uint32_t mx_try_send(struct mx_post *);



/* mx_create_rdma_window.c */
mx_return_t
mx_create_rdma_window(mx_endpoint_t endpoint,
                      mx_segment_t *segment_list,
                      uint32_t segment_count,
                      uint32_t flags,
                      uint32_t *window_handle);

/* mx_rdma_write.c */
mx_return_t
mx_rdma_write(mx_endpoint_t endpoint,
              mx_segment_t *segments_list,
              uint32_t segments_count, 
              mx_endpoint_addr_t destination,
              uint32_t window_handle,
              uint32_t offset,
              void *context,
              mx_request_t *request);


/* tcp.c */
int get_sock_for_address(struct mx_endpoint *, mx_endpoint_addr_t);
struct mx_address_desc *mx_new_address_desc(mx_endpoint_addr_t);
struct mx_address_desc *mx_get_address_desc(struct mx_endpoint *,
                                            mx_endpoint_addr_t);
void mx_destroy_address_desc(struct mx_address_desc* ep);


/* packet.c */
int mx_sock_read(int, void *, uint32_t);
int mx_sock_write(int, void *, uint32_t);
void mx_trash_packet(int , uint32_t);
void mx_close_recv_socket(struct mx_endpoint *,
                          struct mx_address_desc *);
void mx_receive_data(struct mx_endpoint *, struct mx_address_desc *);
void mx_establish_connection(struct mx_endpoint *);

/* receive.c */
struct mx_post *mx_find_recv(struct mx_endpoint *, uint64_t,
                             mx_sr_type_t);

void mx_fulfill_recv(struct mx_post *, struct mx_address_desc *,
                     uint32_t, uint64_t, mx_sr_type_t, mx_request_t);

void mx_queue_msg(struct mx_endpoint *, struct mx_address_desc *,
                  uint64_t, mx_request_t, uint32_t, mx_sr_type_t);
/*
  void mx_queue_msg(struct mx_endpoint *, struct mx_address_desc *,
  uint64_t, uint32_t, mx_post_type_t);
*/
uint32_t mx_match_queued_message(struct mx_post *, mx_sr_type_t);
uint32_t mx_match_queued_barrier(struct mx_post *, mx_sr_type_t);
void mx_destroy_queued_message_list(struct mx_queued_msg *);
void mx_process_barrier(struct mx_endpoint *,
			struct mx_address_desc *ep,
			uint64_t match_info,
			uint32_t len,
			mx_post_type_t type);


void mx_process_issend_ack(struct mx_endpoint *endpoint,
                           struct mx_address_desc *ep);

void mx_process_put_ack(struct mx_endpoint *endpoint,
                        struct mx_address_desc *ep);

/* port.c */
void mx_endpoint_sock_init(struct mx_endpoint *);
void mx_endpoint_sock_insert(struct mx_endpoint *, int);
void mx_endpoint_sock_remove(struct mx_endpoint *, int);
int mx_endpoint_sock_ready(struct mx_endpoint *, int);
int mx_endpoint_sock_bad(struct mx_endpoint *, int);

/* attrib.c */
void mx_free_attr_list(struct mx_endpoint *);
mx_return_t mx_set_dflt_endpoint_attribs(struct mx_endpoint *);

/* rdma.c */
void mx_process_put(mx_endpoint_t endpoint,
                    struct mx_address_desc *ep, 
                    uint32_t data_len,
                    uint32_t rdma_handle,
                    uint32_t rdma_offset,
                    mx_request_t sender_request);
uint32_t mx_add_recv_window_handle(mx_endpoint_t endpoint, 
                                   void *ptr, int len);
void mx_del_recv_window_handle(mx_endpoint_t endpoint, uint32_t index);

void mx_process_get(mx_endpoint_t endpoint,
                    struct mx_address_desc *ep,
                    mx_request_t sender_request);

/* generic doubly linked list macros
 * assumes that an anchor exists whose initial prev and next pointers
 * point to itself */

#define MX_LIST_INIT(p) \
do {                    \
  (p)->next = (p);      \
  (p)->prev = (p);      \
} while (0)

/* t - type
 * v - list to iterate over
 * f - function to call for each node */
#define MX_LIST_APPLY(t,v,f) \
do {                         \
  t *p, *q;                  \
  p = (v)->next;             \
  while (p != v) {           \
    q = p;                   \
    p = p->next;             \
    f(q);                    \
  }                          \
} while (0)

/* insert q _before_ p */
#define MX_LIST_INSERT(p,q) \
do {                        \
  (p)->prev->next = q;      \
  q->prev = (p)->prev;      \
  q->next = p;              \
  (p)->prev = q;            \
} while (0)

/* splice p out of the list */
#define MX_LIST_REMOVE(p)  \
do {                       \
  p->next->prev = p->prev; \
  p->prev->next = p->next; \
} while (0)

/* t - type
 * v - list to iterate over
 * p - position to start from
 * f - expression to test
 * a - arg to pass to f */
#define MX_LIST_FIND(t,v,p,f,a) \
do {                            \
  for(; p != v; p = p->next) {  \
    if (f(p, a)) {              \
      break;                    \
    }                           \
  }                             \
} while (0)


//pthread_mutex_t Mx_lock;
pthread_mutex_t Mx_tx_lock;
pthread_mutex_t Mx_rx_lock;
pthread_mutex_t Mx_po_lock;

int Mx_init_count;
struct mx_endpoint Mx_endpoints;
pthread_t Mx_tx_thread;
pthread_t Mx_rx_thread;
int Mx_main_cancel;
pthread_cond_t Mx_send_cond;

#if 0
void *mx_safe_malloc(size_t size);
void *mx_safe_calloc(size_t nmemb, size_t size);
void mx_safe_free(void *ptr);

#define malloc(x)    mx_safe_malloc(x)
#define calloc(x, y) mx_safe_calloc(x, y)
#define free(x)      mx_safe_free(x)
#endif
